/*
 *  Copyright 2008 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.apache.ibatis.ibator.plugins;

import java.util.ArrayList;
import java.util.List;

import org.apache.ibatis.ibator.api.GeneratedXmlFile;
import org.apache.ibatis.ibator.api.IbatorPluginAdapter;
import org.apache.ibatis.ibator.api.IntrospectedTable;
import org.apache.ibatis.ibator.api.dom.xml.Attribute;
import org.apache.ibatis.ibator.api.dom.xml.Document;
import org.apache.ibatis.ibator.api.dom.xml.XmlElement;
import org.apache.ibatis.ibator.config.TableConfiguration;
import org.apache.ibatis.ibator.generator.ibatis2.XmlConstants;
import org.apache.ibatis.ibator.internal.util.StringUtility;
import org.apache.ibatis.ibator.internal.util.messages.Messages;

/**
 * This plugin generates an SqlMapConfig file containing sqlMap entries for SQL
 * maps generated by ibator. This demonstrates hooking into the ibator code
 * generation lifecycle and generating additional XML files.
 * <p>
 * This plugin accepts three properties:
 * <ul>
 * <li><tt>fileName</tt> (optional) the name of the generated file. this
 * defaults to "SqlMapConfig.xml" if not specified.</li>
 * <li><tt>targetPackage</tt> (required) the name of the package where the file
 * should be placed. Specified like "com.mycompany.sql".</li>
 * <li><tt>targetProject</tt> (required) the name of the project where the file
 * should be placed.</li>
 * </ul>
 * 
 * Note: targetPackage and targetProject follow the same rules as the
 * targetPackage and targetProject values on the sqlMapGenerator configuration
 * element.
 * 
 * @author Jeff Butler
 * 
 */
public class SqlMapConfigPlugin extends IbatorPluginAdapter {

	private List<String> sqlMapFiles;

	public SqlMapConfigPlugin() {
		sqlMapFiles = new ArrayList<String>();
	}

	public boolean validate(List<String> warnings) {
		boolean valid = true;

		if (!StringUtility.stringHasValue(properties
				.getProperty("targetProject"))) { //$NON-NLS-1$
			warnings.add(Messages.getString("Warning.25")); //$NON-NLS-1$
			valid = false;
		}

		if (!StringUtility.stringHasValue(properties
				.getProperty("targetPackage"))) { //$NON-NLS-1$
			warnings.add(Messages.getString("Warning.26")); //$NON-NLS-1$
			valid = false;
		}

		return valid;
	}

	@Override
	public List<GeneratedXmlFile> contextGenerateAdditionalXmlFiles(
			List<IntrospectedTable> tables) {
		Document document = new Document(XmlConstants.SQL_MAP_CONFIG_PUBLIC_ID,
				XmlConstants.SQL_MAP_CONFIG_SYSTEM_ID);

		XmlElement root = new XmlElement("sqlMapConfig"); //$NON-NLS-1$
		document.setRootElement(root);

		XmlElement settings = new XmlElement("settings"); //$NON-NLS-1$
		settings.addAttribute(new Attribute("useStatementNamespaces", "true")); //$NON-NLS-1$ //$NON-NLS-2$
		root.addElement(settings);

		XmlElement sqlMap;
		for (String sqlMapFile : sqlMapFiles) {
			sqlMap = new XmlElement("sqlMap"); //$NON-NLS-1$
			sqlMap.addAttribute(new Attribute("resource", sqlMapFile)); //$NON-NLS-1$
			root.addElement(sqlMap);
		}

		// baisui add
		// String schema = null;
		// for (TableConfiguration tbConfig : this.ibatorContext
		// .getTableConfigurations()) {
		// schema = tbConfig.getSchema();
		// break;
		// }

		// GeneratedXmlFile gxf = new GeneratedXmlFile(document, properties
		//				.getProperty("fileName", "sqlMapConfig.xml"), //$NON-NLS-1$ //$NON-NLS-2$
		//				properties.getProperty("targetPackage"), //$NON-NLS-1$
		//				properties.getProperty("targetProject"), //$NON-NLS-1$
		// false);

		GeneratedXmlFile gxf = new GeneratedXmlFile(
				document,
				properties
						.getProperty(
								"fileName", this.ibatorContext.getSqlMapConfigFileName()), //$NON-NLS-1$ //$NON-NLS-2$
				properties.getProperty("targetPackage"), //$NON-NLS-1$
				properties.getProperty("targetProject"), //$NON-NLS-1$
				false);

		List<GeneratedXmlFile> answer = new ArrayList<GeneratedXmlFile>(1);
		answer.add(gxf);

		return answer;
	}

	private String dbSchema;

	/*
	 * This method collects the name of every SqlMap file generated by ibator in
	 * this context.
	 */
	@Override
	public boolean sqlMapGenerated(GeneratedXmlFile sqlMap,
			IntrospectedTable introspectedTable) {
		StringBuilder sb = new StringBuilder();
		sb.append(sqlMap.getTargetPackage());
		sb.append('.');
		String temp = sb.toString();
		sb.setLength(0);
		sb.append(temp.replace('.', '/'));
		sb.append(sqlMap.getFileName());
		sqlMapFiles.add(sb.toString());

		return true;
	}
}
