/**
 * Project: ibator_koubei
 * <p>
 * File Created at 2009-10-22
 */
package com.koubei.abator;

import org.apache.commons.lang.StringUtils;
import org.apache.ibatis.ibator.api.Ibator;
import org.apache.ibatis.ibator.config.IbatorConfiguration;
import org.apache.ibatis.ibator.config.IbatorContext;
import org.apache.ibatis.ibator.config.xml.IbatorConfigurationParser;
import org.apache.ibatis.ibator.internal.DefaultShellCallback;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class KoubeiIbatorRunner {
    private IbatorContext ibatorContext;
    private final IbatorConfiguration config;
    private List<String> warnings = new ArrayList<String>();

    public static void main(String[] args) throws Exception {

        IUserDefineProperties properties = getDefineProperties();
        KoubeiIbatorRunner runner = new KoubeiIbatorRunner(properties);
        runner.build();
    }

    public KoubeiIbatorRunner(IUserDefineProperties properties) {
        super();
        this.properties = properties;


        try {
            Properties property = this.properties.getProperty();

            createOutputDir(new File(property.getProperty("outputDir")));

            IbatorConfigurationParser cp = new IbatorConfigurationParser(property, warnings);

            config = cp.parseIbatorConfiguration(
                    KoubeiIbatorRunner.class.getResourceAsStream("/com/koubei/abator/ibator.xml"));

            this.ibatorContext = config.getIbatorContext("mysqltables");
            this.ibatorContext.setUserConfig(this.properties);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public IbatorContext getIbatorContext() {
        if (ibatorContext == null) {
            throw new IllegalStateException("ibatorContext can not be null");
        }
        return ibatorContext;
    }

    public void build() throws Exception {

        parseTables(ibatorContext);

        Ibator ibator = new Ibator(config, getShellCallback(), warnings, this);

        ibator.generate(getProgressCallback());
    }

    protected DefaultShellCallback getShellCallback() {
        DefaultShellCallback callback = new DefaultShellCallback(true);
        return callback;
    }

    private void createOutputDir(File div) {
        if (!div.exists()) {
            div.mkdirs();
        }
    }

    protected KoubeiProgressCallback getProgressCallback() {
        return new KoubeiProgressCallback();
    }

    // protected void createOutputDir(File outputDir) {
    // File outputDir = new File("target\\src");
    // if (!outputDir.exists()) {
    // outputDir.mkdirs();
    // }
    // }

    private void parseTables(IbatorContext ibatorContext) {
        ibatorContext.getTableConfigurations().clear();
        // ibatorContext.addTableConfiguration(tc)
        // KoubeiTableConfiguration config = null;
        Set<String> tables = getFullyQualifiedTableNames(ibatorContext);
        for (String t : tables) {
            ibatorContext.addTableConfiguration(new KoubeiTableConfiguration(t, ibatorContext));
        }
    }

    // private static final Pattern p = Pattern.compile("^table\\.(\\w+)$");

    private Set<String> getFullyQualifiedTableNames(IbatorContext ibatorContext) {
        final Set<String> fullyQualifiedTableNames = new HashSet<String>();
        String tables = this.properties.getTables();
        if ("all".equalsIgnoreCase(StringUtils.trimToEmpty(tables))) {
            return ibatorContext.getAllTables();
        }
        StringTokenizer tokenizer = new StringTokenizer(tables, ",");
        while (tokenizer.hasMoreTokens()) {
            fullyQualifiedTableNames.add(tokenizer.nextToken());
        }
        return fullyQualifiedTableNames;
    }

    private final IUserDefineProperties properties;

    public static IUserDefineProperties getDefineProperties() {
        UserDefineProperties properties = null;
        if (properties == null) {
            synchronized (KoubeiIbatorRunner.class) {
                if (properties == null) {
                    File propFile = new File("config.properties");
                    if (!propFile.exists()) {
                        throw new IllegalStateException("config file:" + propFile.getAbsolutePath() + " is not exist");
                    }
                    properties = new UserDefineProperties(System.getProperties());
                    InputStream input = null;
                    try {
                        input = new FileInputStream(propFile);
                        properties.property.load(input);
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    } finally {
                        try {
                            input.close();
                        } catch (Exception e) {
                        }
                    }
                }
            }
        }

        return properties;
    }

    public static class UserDefineProperties extends AdapterUserDefineProperties {
        private final Properties property;

        public UserDefineProperties(Properties property) {
            super();
            this.property = property;
        }

        @Override
        public String getDatabaseName() {
            return this.property.getProperty(KEY_databaseName);
        }

        @Override
        public Properties getProperty() {
            return this.property;
        }

        @Override
        public String getTables() {
            return property.getProperty("tables", "");
        }

        @Override
        public String getOutputfileEncode() {
            return property.getProperty("outputfileEncode", "utf8");
        }

        @Override
        public boolean generateDataSourceConfig() {
            return true;
        }

        /**
         * sqlmapBoolean类型映射启用
         *
         * @return
         */
        @Override
        public boolean isBooleanMapEnable() {
            return "true".equalsIgnoreCase(property.getProperty("enable.sqlmap.boolean.type"));
        }

        @Override
        public boolean isDisableGenerateModifyDAOMethod() {
            return "true".equalsIgnoreCase(property.getProperty("disable.generate.modify.dao.method"));
        }

        private static final Pattern pkg_pattern = Pattern.compile("[a-z]+(\\.[a-z]+)+");

        @Override
        public String getDependencyPackageParentName() {
            String val = property.getProperty("dependency.package.parent.name");
            Matcher matcher = pkg_pattern.matcher(val);
            if (!matcher.matches()) {
                throw new IllegalStateException("val:" + val + " is not match the pattern:" + pkg_pattern);
            }
            return val;
        }

        @Override
        public PojoExtendsClass getPojoExtendsClass() {
            String val = property.getProperty("pojo.extends.class");
            if (StringUtils.isBlank(val)) {
                return null;
            }
            String[] args = StringUtils.split(val, ".");
            if (args.length < 2) {
                throw new IllegalStateException("key 'pojo.extends.class' relevant val is illgeal");
            }
            StringBuffer packageName = new StringBuffer();
            for (int i = 0; i < args.length - 1; i++) {
                packageName.append(args[i]).append(".");
            }
            String className = args[args.length - 1];
            packageName.append(className);
            return new PojoExtendsClass(packageName.toString(), className);
        }

        // "outputfileEncode", "utf8"


    }

    public static class PojoExtendsClass {
        private final String fullClassName;
        private final String className;

        public PojoExtendsClass(String fullClassName, String className) {
            super();
            this.fullClassName = fullClassName;
            this.className = className;
        }

        public String getFullClassName() {
            return this.fullClassName;
        }

        public String getClassName() {
            return this.className;
        }

    }
}
